# Cross-Domain Workflow: Scan Live, Fix Local

Complete walkthrough of scanning a production site and applying fixes in a
local development project.

## Scenario

- **Live site**: `https://example.com` (production)
- **Local dev**: `https://example.test` (Herd)
- **Working directory**: contains the source code for `example.com`
- **Goal**: scan the live site, fix issues in local source files, verify locally

## Step 1: Scan the live site

Run the standard 3-step async scan against the production URL:

```bash
# Start scan
seogent scan https://example.com --quiet --dev
# → { "scan_id": "scan_xyz789", "status": "pending", ... }

# Poll
sleep 15 && seogent status scan_xyz789 --quiet --dev
# Repeat until "completed"

# Fetch results
seogent results scan_xyz789 --quiet --dev
```

## Step 2: Establish domain mapping

Map the scan domain to the local domain:
```
https://example.com  →  https://example.test
```

When processing results, strip the domain and work with paths:
```
https://example.com/about    →  path: /about
https://example.com/services →  path: /services
https://example.com/contact  →  path: /contact
```

## Step 3: Detect project type

Check for signature files in the working directory:

```bash
# Check for common project signatures
ls artisan craft next.config.js astro.config.mjs wp-config.php hugo.toml .eleventy.js 2>/dev/null
```

### Example: Craft CMS detected

Signals found:
- `craft` binary exists → Craft CMS project
- `templates/` directory exists → confirms Craft CMS

File mapping strategy for Craft CMS:
- **Base layout**: `templates/_layout.twig` or `templates/_base.twig`
- **Page templates**: `templates/{path}.twig` or `templates/{path}/index.twig`

### Example: Laravel/Inertia detected

Signals found:
- `artisan` exists → Laravel project
- `app/Http/Middleware/HandleInertiaRequests.php` exists → Inertia variant

File mapping strategy for Laravel/Inertia:
- **Base layout**: `resources/js/Layouts/AppLayout.tsx` (or similar)
- **Page components**: `resources/js/Pages/{Path}.tsx`
- **Routes**: `routes/web.php` → maps URL paths to controllers → controllers reference page components

### Example: Static HTML detected

Signals found:
- `index.html` in root, no framework config files

File mapping strategy:
- **Homepage**: `index.html`
- **Other pages**: `{path}/index.html` or `{path}.html`
- **No shared layout** — site-wide fixes may require editing every HTML file

## Step 4: Resolve source files for issues

### Site-wide issues (fix the base layout)

From scan results — `top_issues`:
```json
{ "issue": "Missing canonical tag", "count": 28 },
{ "issue": "Missing Open Graph tags", "count": 30 }
```

These affect nearly every page → fix in the base layout.

**Craft CMS example**: Edit `templates/_layout.twig`
```twig
{# Add to <head> section #}
<link rel="canonical" href="{{ siteUrl ~ craft.app.request.pathInfo }}">
<meta property="og:title" content="{{ title ?? siteName }}">
<meta property="og:description" content="{{ description ?? '' }}">
<meta property="og:url" content="{{ siteUrl ~ craft.app.request.pathInfo }}">
```

**Laravel/Inertia example**: Edit the root layout's `<Head>` component
```tsx
<Head>
  <link rel="canonical" href={route('current')} />
  <meta property="og:title" content={title} />
  <meta property="og:description" content={description} />
</Head>
```

### Page-specific issues

From scan results — one page has unique issues:
```json
{
  "url": "https://example.com/about",
  "failed_checks": ["Missing meta description", "Heading hierarchy skipped"]
}
```

1. Extract path: `/about`
2. Find source file:
   - **Craft CMS**: look for `templates/about.twig` or `templates/about/index.twig`
   - **Laravel/Inertia**: check `routes/web.php` for `/about` route → find the
     controller → find the Inertia page component
   - **Static HTML**: look for `about/index.html` or `about.html`
3. If path-based lookup fails, search for content:
   ```bash
   # Search for the page's title text to find the right file
   grep -r "About Us" templates/ --include="*.twig" -l
   ```
4. Apply the fix to that specific file

## Step 5: Verify fixes locally

After applying all fixes, re-scan using the **local dev domain**:

```bash
# Verify site-wide fixes on a few representative pages
seogent scan https://example.test/ \
  --urls https://example.test/about \
  --urls https://example.test/services \
  --quiet --dev

# Poll and fetch results as usual
sleep 15 && seogent status <scan_id> --quiet --dev
seogent results <scan_id> --quiet --dev
```

Compare scores before and after:

| Page | Before | After |
|---|---|---|
| `/` | 82 (B) | 95 (A) |
| `/about` | 58 (D) | 88 (B) |
| `/services` | 71 (C) | 92 (A) |

## Step 6: Handle edge cases

### URL path doesn't match a file

Some projects use dynamic routing where URL paths don't directly map to
files. For example, a blog post at `/blog/my-post` might be rendered by a
single template that loads content from a database.

Resolution:
1. Check the routing config (e.g., `routes/web.php`, Craft CMS sections)
2. Identify the template that handles that route pattern
3. For content-level issues (missing description), the fix may be in the
   CMS admin or database, not in a template file
4. Ask the user if the mapping is unclear

### Multiple projects in one directory

If the working directory contains multiple project signatures (e.g., a
monorepo), ask the user which sub-project corresponds to the scanned site.

### Template inheritance chains

Many projects use multi-level template inheritance:
```
_base.twig → _layout.twig → about.twig
```

For `<head>` fixes (canonical, OG tags), find the **highest-level template**
that contains the `<head>` block. For content fixes (headings, alt text),
find the **page-level template**.

## Summary

The cross-domain workflow is:
1. **Scan** the live production URL
2. **Map** result URLs to local paths by stripping the domain
3. **Detect** the project type from signature files
4. **Resolve** source files using project-type-specific mapping
5. **Fix** issues in the local source files
6. **Verify** by re-scanning the local dev URL
